%> @file Spitfire_Arduino.m
%> @brief Enthlt Parameter des Spitfire Ace
%> @brief Enthlt Methoden zur Steuerung des Spitfire Ace am LAT.
%> Der Laser wird ber einen Arduino MEGA angesteuert, der die Emission, sowie
%> den Shutter steuert.
%> Im Programm des Arduino sind verschiedene Befehle hinterlegt, die per
%> serieller Schnittstelle aus dieser Klasse heraus aufgerufen werden.
%
%> Die Leistung wird ber eine drehbare Lambda-Halbe-Platte eingestellt
%> Steuerungsmodul: https://www.thorlabs.com/thorproduct.cfm?partnumber=ELL14
classdef Spitfire_Arduino < Laser
    
    properties
        %Arduino Properties
        %>Serielle Schnittstelle
        comPortArduino;             
        %>Baudrate fr Arduino
        baudRateArduino = 9600;     
        %>Com-Port des Arduino
        ArduinoPort = 1;            
        %>Hier werden die von Arduino ausgegebenen Rckgabewerte gespeichert
        ArduinoMessages;            
        
        %Shutter Properties
        %true = open; Aktueller Zustand des Shutters
        shutter = false;    
        %Winkel in DEG des Servomotors, bei dem der Shutter geffnet ist
        angleOpen = 20;%                 
        %Winkel in DEG des Servomotors, bei dem der Shutter geschlossen ist
        angleClose = 120; %              
        
        %Power Control Properties
        %>Serielle Schnittstelle
        comPortELL14;                       
        %>Verbindungszustand der Leistungssteuerung
        ELL14Connected = false;             
        %>Beudrate der Leistungssteuerung
        baudRateELL14 = 9600;               
        %>Com-Port der Leistungssteuerung
        ELL14Port = 1;                      
        %>Aktueller Winkel der Lambda-Halbe-Platte
        ELL14Angle = 0;                     
        %>Schrittweite der Drehung der Lambda-Halbe-Platte
        ELL14Step = 1; %[]                
        %> Bei "true" wird Warnung bei Verstellen der Leistung ausgegeben
        ELL14SafetyWarning = false;         
        %> 'Warning Message', 'Only at Closed Shutter', 'Only During Power Measurement'
        ELL14SafetyWarningType = 'Warning Message'; 

        %Pulse Picker Properties
        ppDelayOpenLaser = 25;%ms
        ppLaserTriggerWidth = 50;%ms
        ppDelayOpen2ndTrigger =5;%ms
        pp2ndTriggerWidth =100;%ms
    end
    
    methods
        %> @brief Konstruktor mit Standardparametern des Lasers
        function obj = Spitfire_Arduino()            
            obj.wavelength = 800;
            obj.rawBeamDiameter = 5.1;
            obj.aperture = 1000;
        end
        
        %> @brief Herstellen der Verbindung zum Arduino
        function connect(obj)
            
            obj.comPortArduino = serialport(strcat("COM", num2str(obj.ArduinoPort)), obj.baudRateArduino);
            obj.connected = true;
            %Warten, bis Arduino verbunden ist
            pause(1);
            %Setzen der Shutterwinkel
            obj.setShutterOpenAngle();
            pause(1);
            obj.setShutterCloseAngle();
        end

        %> @brief Lschen der Verbindung zum Arduino
        function disconnect(obj)        
            obj.comPortArduino = [];
            obj.connected =false;
            %#muss hier auch Verbindung zum ELL14 gelscht werden?
        end

        %> @brief Einschalten des Lasers per Befehl an den Arduino
        function laserOn(obj)      
            obj.arduinoCommandQuick("LaserOpen");
            obj.emission = true;
        end

        %> @brief Ausschalten des Lasers per Befehl an den Arduino
        function laserOff(obj) 
            obj.arduinoCommandQuick("LaserClose");
            obj.emission = false;
        end

         %> @brief ffnen des Shutters per Befehl an den Arduino
        function shutterOpen(obj)     
            obj.arduinoCommandQuick("ShutterOpen");
            obj.shutter = true;
        end

         %> @brief Schlieen des Shutters per Befehl an den Arduino
        function shutterClose(obj)
            obj.arduinoCommandQuick("ShutterClose");
            obj.shutter = false;
        end

        %> @brief Setzen des Winkels zum ffnen des Shutters per Befehl an den Arduino
        function setShutterOpenAngle(obj)
            obj.arduinoCommand(strcat("ServoPositionOpen ",num2str(obj.angleOpen)));
        end

        %> @brief Setzen des Winkels zum Schlieen des Shutters per Befehl an den Arduino
        function setShutterCloseAngle(obj)  
            obj.arduinoCommand(strcat("ServoPositionClose ",num2str(obj.angleClose)));
        end
        
        %> @brief Einzelpuls aus Laser auskoppeln
        function pulsePicker(obj)  
            obj.arduinoCommandQuick("PulsePicker");
        end
        %Einstellungen Pulse Picker
        %> @brief Setzen des Delays zum Trigger des Lasers
        function setppDelayOpenLaser(obj)  
            obj.arduinoCommand(strcat("ppDelayOpenLaser ",num2str(obj.ppDelayOpenLaser)));
        end
        %> @brief Setzen der Breite des Triggersignals des Lasers
        function setppLaserTriggerWidth(obj)  
            obj.arduinoCommand(strcat("ppLaserTriggerWidth ",num2str(obj.ppLaserTriggerWidth)));
        end
        %> @brief Setzen des Delays zum zweiten Trigger
        function setppDelayOpen2ndTrigger(obj)  
            obj.arduinoCommand(strcat("ppDelayOpen2ndTrigger ",num2str(obj.ppDelayOpen2ndTrigger)));
        end
        %> @brief Setzen der Breite des zweiten Triggersignals
        function setpp2ndTriggerWidth(obj)  
            obj.arduinoCommand(strcat("pp2ndTriggerWidth ",num2str(obj.pp2ndTriggerWidth)));
        end

         %> @brief Einschalten des Lasers fr eine definierte Zeit per Befehl an den Arduino
        function laserTime(obj, time)
            obj.arduinoCommand(strcat("LaserTime ",num2str(time*1000)));
            
        end
        
        %ARDUINO
        %> @brief Schickt Befehl an Arduino
        function text = arduinoCommand(obj, command)
            %Wartet Antwort des Arduinio ab und schreibt sie ins Protokoll
            writeline(obj.comPortArduino,command);
            message = strcat("send: ",command);
            obj.ArduinoMessages = [message obj.ArduinoMessages];
            response = "got: ";
            
            i = 1;
            while (response == "got: ")
                value = readline(obj.comPortArduino);
                response = strcat("got: ",value);
                i = i+1;
                if i>10000
                    msgbox("ERROR - no response from Arduino");
                end
            end
            obj.ArduinoMessages = [strcat(datestr(now), response) obj.ArduinoMessages];
            text = value;
            
            %#prfen, was wirklich zurckgegeben werden muss
        end

        %> @brief Schickt Befehl an den Arduino, ohne diesen zu dokumentieren und ohne auf eine Antwort zu warten
        function arduinoCommandQuick (obj, command)
            writeline(obj.comPortArduino,command);
            
        end
        
        %Methoden ELL14
        %> @brief Stellt Verbindung zur Leistungssteuerung her
        function PowerAdjustmentConnect(obj)
            obj.comPortELL14 = serialport(strcat("COM", num2str(obj.ELL14Port)), obj.baudRateELL14);
            obj.ELL14Connected = true;
        end

        %> @brief Lst Verbindung zur Leistungsssteuerung
        function PowerAdjustmentDisconnect(obj)
            obj.comPortELL14 = [];
            obj.ELL14Connected = false;
        end

        %> @brief Stellt Leistungssteuerung auf absoluten Winkel ein
        function PowerAdjustmentRotate(obj)
            hexstr = dec2hex(single(round(obj.ELL14Angle*398.2)));%398.2 ausgelesener Wert aus ELL-Software
            while length(hexstr)<8
                hexstr = ['0' hexstr];
            end
            code = strcat('0ma',hexstr);
            writeline(obj.comPortELL14,code);
        end

        %> @brief Dreht Lambda-Halbe-Platte um relativen Winkel in positive
        %> Drehrichtung (das bedeutet nicht zwingend auch einen positiven
        %> Leistungsgradienten)
        function PowerPlus(obj)
            obj.ELL14Angle = obj.ELL14Angle + obj.ELL14Step;
            obj.PowerAdjustmentRotate();
        end

        function PowerPlusX(x,obj)
            obj.ELL14Angle = obj.ELL14Angle + obj.ELL14Step*x;
            obj.PowerAdjustmentRotate();
        end
        
        %> @brief Dreht Lambda-Halbe-Platte um relativen Winkel in negative
        %> Drehrichtung (das bedeutet nicht zwingend auch einen negativen
        %> Leistungsgradienten)
        function PowerMinus(obj)
            obj.ELL14Angle = obj.ELL14Angle - obj.ELL14Step;
            obj.PowerAdjustmentRotate();
        end

        function PowerMinusX(x,obj)
            obj.ELL14Angle = obj.ELL14Angle - obj.ELL14Step*x;
            obj.PowerAdjustmentRotate();
        end

        %> @brief Warnmeldung bei Leistungsnderung in bestimmten Betriebszustnden
        function value = powerAdjustmentWarning(obj)
            if(obj.ELL14SafetyWarning)
                if(obj.ELL14SafetyWarningType == "Warning Message")
                    %Dialog ffnen, ob Leistung verstellt werden soll
                    answer = questdlg('This will change the Power. Are you sure that the beam will not hit critical components?', ...
                        'Warning', ...
                        'Yes','No --> Cancel');
                    % Handle response
                    switch answer
                        case 'Yes'
                            value = true;
                        case 'No --> Cancel'
                            value = false;
                    end
                elseif(obj.ELL14SafetyWarningType == "Only at Closed Shutter")
                    %Leistungsnderung nur erlauben, wenn Shutter
                    %geschlossen ist
                    if(obj.connected && obj.shutter == false)
                        value = true;
                    else
                        value = false;
                    end
                elseif(obj.ELL14SafetyWarningType == "Only During Power Measurement")
                    %Leistungsnderung nur erlauben, wenn Leistung auf
                    %Powermeter angezeigt wird
                    if(Powermeter.getPower() > 0.002)
                        value = true;
                    else
                        value = false;
                    end
                end
                
            else
                value = true;
            end
        end
    end
end

