
%> @brief Hologramme werden als Objekte vom Typ Hologram dargestellt.
%> Objekte vom Typ Hologram enthalten:
%> <p>&rarr; Allgemeine Hologrammparameter wie Auflsung, ...<br>
%> &rarr; Die Parameter der verwendeten Hologrammalgorithmen<br>
%> &rarr; Das Hologramm, sowie Informationen ber die Input-
%> /Output-Amplitude in Form von Graustufenmatrizen</p>
%> Auerdem sind in der Klasse die Methoden zur Berechnung der
%> Hologramme enthalten
classdef Hologram < handle

    properties (Access = public)

        %> Verhalten
        %>Bei "true" wird whrend der Berechnung der Berechnungsfortschritt angezeigt
        displayProgress = true;
        %>Bei "true" wird das Bild nach der Berechnung direkt an den verbunden SLM bertragen
        send2SLMWhenFinished = false;
        %>Bei "true" wird das Bild nach der Berechnung im Output-Ordner gespeichert
        saveImageWhenFinished = false;
        saveImageFileFormat = 'png';%'png' oder 'tiff'
        saveInputAmplitude = false;
        saveOutputAmplitude = false;
        saveOutputAmplitudeIntensityMap = false;
        intensityMapLogScale = false;
        saveInputPhase = true;
        saveOutputPhase = false;
        createSTL = false;

        %>STL konfigurieren
        STL_base =1;    %mm
        STL_height=1;     %mm
        STL_invert = false;
        STL_mode = 'surface_interpolation'; %'surface_interpolation', 'surface_interpolation_x4' oder 'squares'
        interpol_x4_margin=0;%[% der Pixelgre]

        %>Auflsung der Berechnung
        pixelX;
        pixelY;
        %>Abstand der Pixel
        pixelPitch;
        %>Phase diskretiesieren
        discretePhase = false;
        %>Rundungsmethode
        phaseRoundMethod = 'next';%'next','down' oder 'up'
        %>Anzahl der Graustufen
        grayLevels;
        %>Linerarer Faktor zur Kalibrierung des Ablenkwinkels von bspw. dargestellten Blaze-Gittern
        angleCalibrationFactor;
        %>Linerarer Faktor zur Kalibrierung der Fokusverschiebung durch dargestellte Linsen
        lensCalibrationFactor;
        %>Wellenlnge, fr die das Hologramm berechnet wurde in nm
        wavelength;
        %>Modulationstiefe: Maximales Vielfaches der Wellenlnge, das von SLM/DOE dargestellt werden kann
        modulationDepth = 1;
        %>'original' oder 'stretch' #wofr war das nochmal?
        adaptSize = 'original';

        %>Bilder
        %>Matrix mit Phaseninformation des Hologramms
        inputPhase;
        %>Matrix mit relativer Intensittsamplitude des Inputstrahls
        inputAmplitude;
        %>Phase des Strahls im Fernfeld (nach FFT)
        farFieldPhase;
        %>Berechnete Intensittsverteilung im Fernfeld (nach FFT)
        farFieldAmplitude;
        %>Anzeigedauer des Hologramms
        displayDuration;

        %>Dateispeicherpfad fr Hologramm
        savePath;
        %>Dateiname fr Hologramm
        saveName = "";
        %>Fortlaufende Nummer fr Dateinamen
        saveNo=0;

        %>Eingangsstrahl
        %>Sourse = 'calculated', 'fromFile', 'measured'   %Gibt an, woher die Intensittsverteilung fr den Eingangsstrahl genommen werden sollen
        inputBeam_source = 'calculated';
        %> Durchmesser in mm des gaussfrmigen Eingangsstrahls (Nur bei inputBeam_source = 'calculated' relevant)
        inputBeamDiameter=1;%[mm]

        %Algorithmen:
        %Hier werden die Eingabeparameter aller zur Auswahl stehender
        %Algorithmen gespeichert
        %Der 'add####'-Parameter gibt jeweils an, ob der Algorithmus bei
        %der Berechnung ausgefhrt wird.

        %>Prisma und Linse
        addPrismAndLens = false;
        %>Ablenkungswinkel in X-Richtung
        phiX = 0;
        %>Ablenkungswinkel in Y-Richtung
        phiY = 0;
        %>Z-Verschiebung durch Linse
        z = 0;

        %>Multispot
        addMultispot = false;
        %>Koordinatenmatrix der einzelnen Spots mit Parametern (phiX,phiY,Z,I). I ist die Gewichtung der einzelnen Spots.
        spotList = zeros(8,4);

        %>Axikon
        addAxicon = false;
        %>Axikonwinkel
        alpha = 0;
        %>Verschiebung des Mittelpunkts des Axikons in X-Richtung
        ax_x0 = 0;
        %>Verschiebung des Mittelpunkts des Axikons in Y-Richtung
        ax_y0 = 0;

        %>Multi-Axicon
        addMultiAxicon = false;
        %>Koordinatenmatrix der einzelnen Spots mit Parametern (phiX,phiY,Z,alpha,I). I ist die Gewichtung der einzelnen Spots.
        spotListAxicon = zeros(8,5);

        %>Vortex
        addVortex = false;
        %>Topologische Ordnung des Vortexbeams
        charge = 1;
        %>Verschiebung des Mittelpunkts des Vortexmusters in X-Richtung
        vo_x0 = 0;
        %>Verschiebung des Mittelpunkts des Vortexmusters in Y-Richtung
        vo_y0 = 0;

        %>GerchbergSaxton
        addGerchbergSaxton = false;
        %>Anzahl der Iterationen
        gs_iteration = 1;
        %>'random' oder 'quadratic' Startbild fr den Algorithmus: Zufallsbild oder quadratische Phase
        gs_startPhase = 'random';
        %>Streckfaktor der Parabel bei quadratischer Startphase
        gs_quadraticParameter = 1;

        %>Dateipfad zu dem Bild, welches dargestellt werden soll
        gs_image;
        %>'stretch', 'original', 'repeat' Anpassung des Bildes an die Hologrammauflsung
        gs_resizeMode = 'stretch';
        %>'grayValues' oder 'binary'    Konvertierung des darzustelenden Bildes in Graustufen oder Schwarz-Wei
        gs_color = 'grayValues';
        %>Schwellwert fr die Schwarz-Wei-Konvertierung
        gs_limitGrayLevel = 128;
        %>Invertierung der Farben
        gs_invertColors = false;
        %>Array, in dem bei jedem Iterationsschritt die aktuelle Abweichung des Hologramms zum Target-Bild angegeben wird
        gs_convergence;
        %>Bei "true", wird nach der Berechnung der Fehler ber die Iterationen als Plot ausgegeben
        gs_plot = false;


        %G>enetischer Algorithmus
        addGeneticAlgorithm = false;
        %>'random' oder 'gerchbergSaxton'  Startbild fr den Algorithmus: Zufallsbild oder es wird vorher der Gerchberg-Saxton-Algorithmus ausgefhrt, um Startbilder zu generieren
        ga_startPhase = 'random';
        %>Population = Anzahl der Bilder, die in jedem Schritt getestet werden
        ga_population=1;
        %>in %    Wahrscheinlichkeit, mit der sich ein Bit in einem Iterationsschritt verndert
        ga_mutationProb=1;
        %>in % Anteil der Bilder, die fr den nchsten Iterationsschritt behalten wird
        ga_elite=0;
        %>Anzahl der Iterationen
        ga_iteration=1;
        %>Farbtiefe &rarr; entspricht log2(Graustufen) des Hologramms
        ga_imageDepth = 8;
        %>'binary', 'gray' Binre Codierung der Graustufen oder Nutzung des 'Greycode' (Binr ist deutlich schneller)
        ga_bitCode = 'binary';

        %>Dateipfad zu dem Bild, welches dargestellt werden soll
        ga_image;%#s.o.
        %>'stretch', 'original', 'repeat' Anpassung des Bildes an die Hologrammauflsung
        ga_resizeMode = 'original';
        %>'grayValues' oder 'binary'    Konvertierung des darzustelenden Bildes in Graustufen oder Schwarz-Wei
        ga_color = 'grayValues';
        %>Schwellwert fr die Schwarz-Wei-Konvertierung
        ga_limitGrayLevel = 128;
        %>Invertierung der Farben
        ga_invertColors = false;
        %>'calculated' oder 'camera' %Auswahl der Methode zur Evaluation der Bilder. Bei 'calculated' wird die Intensitt mittels FFT simuliert, bei 'camera' wird es mit der Kamera aufgenommen
        ga_evaluation = 'calculated';
        %>Array, in dem bei jedem Iterationsschritt die aktuelle Abweichung des Hologramms zum Target-Bild angegeben wird
        ga_convergence = 0;
        %>Bei "true", wird nach der Berechnung der Fehler ber die Iterationen als Plot ausgegeben
        ga_plot = false;

        %>Zufallsphase
        addRandomPhase = false;
        %>Prozentsatz der Phase, der auf das Hologramm aufaddiert wird
        maximumRandomPhase = 0; %[%]

        %>Zernike
        addZernike = false;
        %>Das Array enthlt die Gewichtungsfaktoren der Zernikepolynome
        weighting = zeros(13);

        % Beginn Damian 2023-11-27
        %> Stitching
        addStitching = false;
        %> Kantenlnge der Optik
        diameterOptic = 3.5;%mm
        %> Kantenlnge Segment
        gratingspacing = 500;%m
        %> Abstand zwischen den Segmenten
        barwidth = 3.7;%m
% Ende Damian 2023-11-27

        %>Ringlochblende
        addAperture = false;
        %>Innerer Radius der Blende in mm
        r_i=0; %[mm]
        %>uerer Radius der Blende in mm
        r_a=1000; %[mm]

        %>Benutzerdefinierte Modifikation des Holograms
        addDevelopmentTools = false;

    end



    methods
        %> @brief Konstruktor:
        %> Wenn SLM verbunden ist, werden dessen Werte fr die Auflsung
        %> etc. fr das Hologramm bernommen
        %> Wenn Laser verbunden ist, wird dessen Wellenlnge bernommen
        function obj = Hologram()

            %Standardspeicherort fr Hologrammbilder
            strcat(evalin('base','matlabPath'),'\data\HologramImages');
            obj.savePath = strcat(evalin('base', 'path'), 'HologramImages');
            try
                obj.pixelX = evalin('base', 'SLM.pixelX');
                obj.pixelY = evalin('base', 'SLM.pixelY');
                obj.pixelPitch = evalin('base', 'SLM.pixelPitch');
                obj.angleCalibrationFactor =  evalin('base', 'SLM.angleCalibrationFactor');
                obj.lensCalibrationFactor =  evalin('base', 'SLM.lensCalibrationFactor');
            catch
                disp('Hologram size not defined!');
            end

            try
                obj.wavelength =  evalin('base', 'Laser.wavelength;');
                obj.inputBeamDiameter =  evalin('base', 'Laser.rawBeamDiameter;');
            catch
                disp('Wavelength not defined!');
            end
        end

        %> @brief Gibt den Abstand jedes Pixels des Holograms zum Mittelpunkt des Holograms
        function matrix = distanceFromCenter(obj)

            matrix=zeros(obj.pixelX,obj.pixelY);
            for ki=1:obj.pixelX
                for kj=1:obj.pixelY
                    % Abstand zum Bildmittelpunkt
                    dx= obj.pixelPitch*(-obj.pixelX*0.5-0.5+ki);
                    dy= obj.pixelPitch*(-obj.pixelY*0.5-0.5+kj);
                    matrix(ki,kj) = sqrt((dx).^2+(dy).^2);
                end
            end
        end

        %> @brief Gibt Matrix mit Abstand jedes Pixels des Holograms zum Punkt x0,y0.
        function matrix = distanceFromPoint(obj,x0,y0)

            matrix=zeros(obj.pixelX,obj.pixelY);
            for ki=1:obj.pixelX
                for kj=1:obj.pixelY
                    %> Abstand zum Bildmittelpunkt
                    dx= obj.pixelPitch*(-obj.pixelX*0.5-0.5+ki)-x0;
                    dy= obj.pixelPitch*(-obj.pixelY*0.5-0.5+kj)-y0;
                    matrix(ki,kj) = sqrt((dx).^2+(dy).^2);
                end
            end
        end

        %> @brief Gibt zu jedem Pixel des Holograms des Winkel bezogen auf den Punkt x0,y0
        function [array] = angle(obj,x0, y0)
            %#auf Bug prfen!
            %ALLES IN mm!
            array=zeros(obj.pixelX,obj.pixelY);
            for ki=1:obj.pixelX
                for kj=1:obj.pixelY
                    %Abstand zum Bildmittelpunkt
                    dx= obj.pixelPitch*(-obj.pixelX*0.5-0.5+ki)-x0;
                    dy= obj.pixelPitch*(-obj.pixelY*0.5-0.5+kj)-y0;
                    if(dx>0)
                        array(ki,kj) = atand(dy/dx)+180;
                    else
                        array(ki,kj) = atand(dy/dx);
                    end
                end
            end
        end

        %> @brief Gibt Bild des Eingangsstrahls aus
        function [image] = beamImage(obj, source, name)

            %Source = 'calculated', 'fromFile', 'measured'
            %Name = Strahlform bei Calculated source, Dateiname bei
            %fromFile
            if(source == "calculated") %Berechnung eines idealen Gaussstrahls
                if(name == "gauss")
                    d0 = evalin('base', 'Laser.rawBeamDiameter;');
                    r=obj.distanceFromCenter();
                    image = gaussmf(r,[d0/4 0]);
                    image = Aperture(image, 0,evalin('base', 'Laser.aperture;')/2);
                else
                    disp("Error, unknown beam shape.");
                end
            elseif(source == "fromFile") %Nutzung eines zuvor aufgenommenen/berechneten Strahlprofils
                disp("#METHODE NOCH NICHT IMPLEMENTIERT");
            elseif(source == "measured") %Aufnahme eines Strahlprofils mit der Kamera
                disp("#METHODE NOCH NICHT IMPLEMENTIERT");
            end

        end

        %> @brief Hauptfunktion zur Hologrammberechnung.
        %>Die Methode fhrt alle ausgewhlten Algorithmen der Reihe nach
        %>aus und addiert die berechneten Hologramme auf.
        %>Die Matrizen zu Eingangs- und Ausgangsstrahl werden in den
        %>entsprechenden Properties gespeichert.
        function hologram = calculate (obj)
            
            disp('Start Calculation...');
            %spter Fortschritt ausgeben
            tic;

            %Schritt 0: Speicherplatz reservieren
            hologram = zeros(obj.pixelX,obj.pixelY); %Das Hologramm ist eine Matrix aus Komplexen Zahlen, bei denen der Betrag die Intensitt und der Winkel die Phase angibt


            %Schritt 1: Hologramm berechnen
            if(obj.addPrismAndLens)
                obj.progress('Calculating Prism & Lens');
                hologram = hologram + angle(Prism_and_Lens(obj.phiX,obj.phiY,obj.z));
            end
            if(obj.addMultispot)
                obj.progress('Calculating Multispot');
                hologram = hologram + angle(Multispot(obj.spotList));
            end
            if(obj.addAxicon)
                obj.progress('Calculating Axicon');
                hologram = hologram + angle(Axicon_Hologram(obj.alpha,obj.ax_x0,obj.ax_y0));
            end
            if(obj.addMultiAxicon)
                obj.progress('Calculating Multi-Axicon');
                hologram = hologram + angle(MultiAxicon(obj.spotListAxicon));
            end
            if(obj.addVortex)
                obj.progress('Calculating Vortex');
                hologram = hologram + angle(Vortex_Hologramm(obj.charge, obj.vo_x0, obj.vo_y0));
            end
            if(obj.addRandomPhase)
                obj.progress('Calculating RandomPhase');
                hologram = hologram + angle(Random_Phase(obj.maximumRandomPhase));
            end
            if(obj.addZernike)
                obj.progress('Calculating Zernike Polynoms');
                hologram = hologram + angle(Zernike(obj.weighting));
            end
            if(obj.addGerchbergSaxton)
                obj.progress('Calculating Gerchberg & Saxton Algorithm');
                hologram = hologram + angle(Gerchberg_Saxton());
            end
            if(obj.addGeneticAlgorithm)
                obj.progress('Calculating Genetic Algorithm');
                hologram = hologram + angle(Genetic_Algorithm());
            end
% Damian Beginn 2023-11-27
            if(obj.addStitching)
                obj.progress('Calculating Stitching');
                hologram = Stitching(hologram,obj.barwidth,obj.gratingspacing);
            end
% Damian Ende
            if(obj.addAperture)
                obj.progress('Calculating Aperture');
                hologram = Aperture(hologram,obj.r_i,obj.r_a);
            end
            if(obj.addDevelopmentTools)%Bisher nicht in GUI hinterlegt
                obj.progress('Calculating Development Tool Modifications');
                hologram = Modification(hologram);
            end
            
            %Zeit
            obj.progress('Hologram generated');
            
            %Schritt 1,5: Diskretisieren der Phase
            if(obj.discretePhase)
               %Auf Bereich 0...max Graylevel skalieren
               hologram=(hologram)./(2*pi).*obj.grayLevels;
               %Runden
               if(obj.phaseRoundMethod == "next")
                   hologram=round(hologram);
               elseif(obj.phaseRoundMethod == "down")
                   hologram=floor(hologram);
               elseif(obj.phaseRoundMethod == "up")
                   hologram=ceil(hologram);
               else
                   disp("Error, unknown rounding method - no discretizing done.");
               end
               %Zurckskalieren
               hologram=hologram./obj.grayLevels.*(2*pi);
               obj.progress('Phase discretized');
            end


            assignin('base',"phase",hologram);%#nur zum testen
            %Schritt 2: Variablen speichern und fr GUI vorbereiten
            hologramPlane = abs(obj.beamImage(obj.inputBeam_source, "gauss")).*exp(1i*(hologram.*obj.modulationDepth));%##gauss ggf. spter noch ndern
            imagePlane = Hologram_FFT(hologramPlane);%Berechnung der Fernfeldintensittsverteilung mittels FFT
            %Speichern der Ergebnisse als Graustufenmatrix
            obj.inputPhase = Phase_Image(hologramPlane);
            obj.inputAmplitude = Amplitude_Image(hologramPlane);
            obj.farFieldPhase = Phase_Image(imagePlane);
            obj.farFieldAmplitude = Amplitude_Image(imagePlane);

            %Schritt 3: An SLM senden
            if(obj.send2SLMWhenFinished)
                SLM = evalin('base', 'SLM;');
                SLM.sendImage(obj.inputPhase);
            end

            %Schritt 4: Bild ggf. Speichern
            if(obj.saveImageWhenFinished)
                obj.progress('Saving images...');
                obj.saveImages();
            end

            %Zeit
            obj.progress('Calculation finished');
            %figure();
            %imshow(Phase_Image(hologramPlane));
        end

        %> @brief Speichert Phase und Amplitude von Hologramm- und Bildebene als Bilddatei ab
        function saveImages (obj)


            obj.saveNo = obj.saveNo + 1;

            if(obj.saveInputPhase)
                pathInputPhase = strcat(obj.savePath,'\','inputPhase_',obj.saveName,'_',num2str(obj.saveNo));
                obj.saveImageFile(obj.inputPhase,pathInputPhase);
            end
            if(obj.saveInputAmplitude)
                pathInputAmplitude = strcat(obj.savePath,'\','inputAmplitude_',obj.saveName,'_',num2str(obj.saveNo));
                obj.saveImageFile(obj.inputAmplitude,pathInputAmplitude);
            end
            if(obj.saveOutputPhase)
                pathFarfieldPhase = strcat(obj.savePath,'\','farFieldPhase_',obj.saveName,'_',num2str(obj.saveNo));
                obj.saveImageFile(obj.farFieldPhase,pathFarfieldPhase);
            end
            if(obj.saveOutputAmplitude)
                pathFarfieldAmplitude = strcat(obj.savePath,'\','farFieldAmplitude_',obj.saveName,'_',num2str(obj.saveNo));
                obj.saveImageFile(obj.farFieldAmplitude,pathFarfieldAmplitude);
            end
            if(obj.saveOutputAmplitudeIntensityMap)
                pathFarfieldAmplitudeIntensityMap = strcat(obj.savePath,'\','farFieldAmplitudeIntensityMap_',obj.saveName,'_',num2str(obj.saveNo),'.png');
                a=figure();
                imagesc(obj.farFieldAmplitude);
                set(gca,'ColorScale','log');
                % Create colorbar
                colorbar;
                saveas(a,pathFarfieldAmplitudeIntensityMap,'png');
            end
            if(obj.createSTL)
                obj.saveSTL();
            end
            
        end

        

        function saveSTL(obj)
            pathSTL = strcat(obj.savePath,'\',obj.saveName,'STL_Model',obj.saveName,'_',num2str(obj.saveNo),'.stl');
            if(obj.STL_invert)
                image = max(max(obj.inputPhase))-obj.inputPhase;

            else
                image = obj.inputPhase;
            end
            if(obj.STL_mode=="surface_interpolation")
                matrix2triangulation_interpol(pathSTL,image,obj.pixelPitch,obj.STL_height,obj.STL_base);
            elseif(obj.STL_mode=="squares")
                matrix2triangulation_square(pathSTL,image,obj.pixelPitch,obj.STL_height,obj.STL_base);
            elseif(obj.STL_mode=="surface_interpolation_x4")
                matrix2triangulation_interpol_x4(pathSTL,image,obj.pixelPitch,obj.STL_height,obj.STL_base,obj.interpol_x4_margin);
            else
                disp("ERROR - Unknown STL mode");
            end

        end

        %Gibt den Fortschritt der Berechnung im Command Window von Matlab aus

        function progress(obj,message)
            if(obj.displayProgress)
                disp(strcat(num2str(toc/60),' min: ',message))
            end
        end
    end

    methods (Static)

        %> @brief Funktion zur automatisierten Berechnung mehrerer Hologramme
        %>Die Hologramme werden als 1-dimensionales Array an die Methode
        %>bergeben
        function calc(list)


            storage = evalin('base','Hologram'); %Das aktuell in der Workspace gespeicherte Hologramm wird hiermit vorm berschreiben geschtzt
            for n = 1:1:size(list,1)
                %Die einzelnen Hologramme in der Liste werden nacheinander
                %berechnet
                assignin('base','Hologram',list(n))
                evalin('base','Hologram.calculate();');
            end
            assignin('base','Hologram',storage); %Das zuvor gespeicherte Hologramm wird wieder in den Workspace gesetzt
        end

        function saveImageFile(file, path)
            if(evalin('base','Hologram.saveImageFileFormat=="png";'))
                path=strcat(path,'.png');
                imwrite(file,path);
            elseif(evalin('base','Hologram.saveImageFileFormat=="tiff";'))
                path=strcat(path,'.tiff');
                create_tiff(file,evalin('base','Hologram.grayLevels;'),path)
            else
                disp("Image Format error: no images saved");
            end
        end

        function sayHi()
            disp('Hi')
        end
    end
end


